% This file contains the MATLAB code for the paper entitled "A Bayesian 
% Approach to Developing a Stochastic Mortality Model for China" written by
% Johnny S.-H. Li, Kenneth Q. Zhou, Xiaobai Zhu, Wai-Sum Chan, and Felix 
% Wai-Hon Chan.

%% Modeling Inputs/Options
polya = 1;  % polya=0 for non-parametric alpha; polya=1 for Cubic B-spline alpha
polyb = 1;  % polyb=0 for non-parametric beta;  polyb=1 for Cubic B-spline beta
sample = 1; % sample=0 for one sigma2y for all y; sample=1 for three sigma2y's for census, 1% and 0.1% years

% Number of samples
numskip = 100;    % how many samples are discarded before an effective sample saved
numburn = 500;  % burn-in period
numsam = 5000;  % number of effective samples
numpred = 35;   % prediction period
numiter = numsam*numskip+numburn; % total iterations required

% Knots for cubic spline
knots = 6:5:66;           % knots (age) for alpha
knots_a = 6:5:66;         % knots (age) for beta
numknots = length(knots); % the number of knots

% Set generator seed
seed = 2017;
rng(seed);

%% Data construction
% Read data
m_org = csvread('Data.csv');
y_org = log(m_org);

% miss_mat determines whether the data point is missing in the original chinese data set
miss_mat = isnan(m_org);
[miss_row,miss_col] = find(isnan(m_org)); 

% data_mat determines whether the data point will be used in parameter estimation. The data point can be either observed or imputated.
data_mat = ~isnan(m_org); 
data_mat(1:80,14:size(m_org,2)) = true; 
[data_row,data_col] = find(data_mat==1);

% impu_mat determines whether the data point will be imputated
impu_mat = miss_mat & data_mat;
[impu_row,impu_col] = find(impu_mat==1);

% Specify the type of data for each year
year_miss = [2:5 7:8 10:13];
year_census = [1 9 20 30];
year_sample1 = [6 15 25];
year_sample01 = [14 16:19 21:24 26:29 31:34]; 
    
% Specify the sample period and age range of the data
age_start = 1;
age_end = 100;
numage = 100;
numyear = 34;

%% Initial settings for MCMC estimation
% Allocate spaces for parameters
alphaa = zeros(numage,numiter);
betaa = zeros(numage,numiter);
kappa = zeros(numyear,numiter);
mu = zeros(numiter,1); 
sigma2 = ones(numiter,1);
s2 = ones(numyear,numiter);

% Set initial values
alphaa(:,1) = -5;
betaa(:,1) = 0.01;
kappa(:,1) = -10;
mu(1) = -1;
s2(:,1) = 0.05;
sigma2(1) = 40;
y_bayes = y_org;

% Construct cubic B-spline for beta
if polyb == 1
    polyx = zeros(numage,4+numknots);
    for i=1:4
        polyx(:,i) = ((log((age_start:age_end)')).^(i-1));
    end
    for i=5:5+numknots-1
        polyx(:,i) = max(((log((age_start:age_end)') - log(knots(i-4)))).^3,0);
    end
    numdeg = 3+numknots; 
end

% Construct cubic B-Spline for alpha
if polya ==1
    polyx_a = zeros(numage,4+length(knots_a));
    for i=1:4
        polyx_a(:,i) = ((log((age_start:age_end)')).^(i-1));
    end
    for i=5:4+length(knots_a)
        polyx_a(:,i) = max(((log((age_start:age_end)') - log(knots_a(i-4)))).^3,0);
    end
end

%% MCMC Sampling
for n = 2:numiter
    
    % Display the number of the current iteration
    if floor(n/100) == n/100
        disp(n)
    end
    
    %% Imputation for sing celss
    y_mu = zeros(1,length(impu_row));
    y_sim = mvnrnd(zeros(1,length(impu_row)),ones(1,length(impu_row)));
    for i = 1 : length(impu_row)
        y_mu(i) = alphaa(impu_row(i),n-1)+betaa(impu_row(i),n-1)*kappa(impu_col(i),n-1);
        y_bayes(impu_row(i),impu_col(i)) = y_mu(i)+y_sim(i)*sqrt(s2(impu_col(i),n-1));
    end
    
    %% Kalman Filter: updating and predicting procedure
    % Notation are the same as in the paper
    k_t = zeros(numage+1,numyear); % kappa_{x,t}
    p_t = zeros(numage+1,numyear); % p_{x,t}
    eta = zeros(numage,numyear); % eta_{x,t}: prediction error
    g = zeros(numage,numyear); % g_{x,t}: Kalman gain
    k = zeros(numyear,1); % kappa_t
    p = zeros(numyear,1); % p_t
    
    % Diffuse initial value for kalman filter
    p(1) = 1/(betaa(:,n-1)'/s2(1,n-1)*betaa(:,n-1));
    k(1) = betaa(:,n-1)'/s2(1,n-1)*(y_bayes(:,1)-alphaa(:,n-1))*p(1);
    k_t(:,1) = k(1);
    p_t(:,1) = p(1);
    
    % Sequential algorithm
    for t = 2:numyear
        k_t(1,t) = mu(n-1) + k_t(numage+1,t-1); %kappa(t|t-1)
        p_t(1,t) = p_t(numage+1,t-1)+sigma2(n-1);            %P(t|t-1)
        if sum(data_mat(:,t))>0 %if there is data for year t
            for j = 1:numage
                if data_mat(j,t) == 0 %if data is missing
                    k_t(j+1,t) = k_t(j,t);
                    p_t(j+1,t) = p_t(j,t);
                else
                    g(j,t) = p_t(j,t)*betaa(j,n-1)/(betaa(j,n-1)^2*p_t(j,t) + s2(t,n-1));
                    eta(j,t) = y_bayes(j,t)-alphaa(j,n-1) - betaa(j,n-1)*k_t(j,t);
                    k_t(j+1,t) = k_t(j,t) + g(j,t)*eta(j,t);
                    p_t(j+1,t) = p_t(j,t) - g(j,t)*betaa(j,n-1)*p_t(j,t);
                end
            end
        else %if there is NO data for year t
            p_t(2:numage+1,t) = p_t(1,t);
            k_t(2:numage+1,t) = k_t(1,t);
        end
        p(t) = p_t(numage+1,t); %P(t,t)
        k(t) = k_t(numage+1,t); %kappa(t,t)
    end

    %% Kalman Filter: sampling procedure
    h = zeros(numyear,1); %E(k(t|t,k(t+1))) = k(t|t,k(t+1))
    H = zeros(numyear,1); %Var(k(t|t,k(t+1))) = P(t|t,k(t+1))
    
    h(numyear) = k(numyear);
    H(numyear) = p(numyear);
    kappa(numyear,n) = normrnd(h(numyear), sqrt(H(numyear)));
    for i=numyear-1:-1:1
        h(i) = k(i)+p(i)/(p(i)+sigma2(n-1))*(kappa(i+1,n)-k(i)-mu(n-1));
        H(i) = p(i)-p(i)^2/(p(i)+sigma2(n-1));
        kappa(i,n) = normrnd(h(i),sqrt(H(i)));
    end
            
    %% Gibbs Sampling for other parameters
    % s^2_t
    if sample == 1 %if sigma_y is different for different years
        sumsum = zeros(3,1);
        for i = 1 : length(data_row)
            if any(data_col(i)==[year_census year_miss])
                sumsum(1) = sumsum(1) + (y_bayes(data_row(i),data_col(i)) ...
                    - alphaa(data_row(i),n-1) - betaa(data_row(i),n-1)*kappa(data_col(i),n))^2;
            elseif any(data_col(i)==year_sample1)
                sumsum(2) = sumsum(2) + (y_bayes(data_row(i),data_col(i)) ...
                    - alphaa(data_row(i),n-1) - betaa(data_row(i),n-1)*kappa(data_col(i),n))^2;
            elseif any(data_col(i)==year_sample01)
                sumsum(3) = sumsum(3) + (y_bayes(data_row(i),data_col(i)) ...
                    - alphaa(data_row(i),n-1) - betaa(data_row(i),n-1)*kappa(data_col(i),n))^2;
            end
        end
        s2([year_census year_miss],n) = 1/gamrnd(sum(sum(data_col==[year_census year_miss]))/2,2/sumsum(1));
        s2(year_sample1,n) = 1/gamrnd(sum(sum(data_col==year_sample1))/2,2/sumsum(2));
        s2(year_sample01,n) = 1/gamrnd(sum(sum(data_col==year_sample01))/2,2/sumsum(3));
    elseif sample == 0 %if sigma_y is same for all years
        sumsum = 0;
        for i = 1 : length(data_row)
            sumsum = sumsum + (y_bayes(data_row(i),data_col(i)) ...
                - alphaa(data_row(i),n-1) - betaa(data_row(i),n-1)*kappa(data_col(i),n))^2;
        end
        s2(:,n) = 1/gamrnd(length(data_row)/2,2/sumsum);
    end
    
    % alpha
    if polya == 1
        % if alpha is in Cubic Spline.
        mu_tmp = 0;
        sigmainv_tmp = 0;
        for i = 1 : length(data_row)
            sample_varinv_tmp = 1/s2(data_col(i),n);
            mu_tmp = mu_tmp + polyx_a(data_row(i),:) ...
                *(y_bayes(data_row(i),data_col(i))-betaa(data_row(i),n-1)*kappa(data_col(i),n))*sample_varinv_tmp;
            sigmainv_tmp = sigmainv_tmp + polyx_a(data_row(i),:)'*polyx_a(data_row(i),:)*sample_varinv_tmp;
        end
       [v,d] = eig(sigmainv_tmp);
       sigma_alpha = v*diag(1./diag(d))*v';
       mu_alpha = sigma_alpha*mu_tmp';
       alphaa(:,n) = polyx_a*(mvnrnd(mu_alpha,sigma_alpha))';
    elseif polya == 0
        % if alpha is not smoothed.
        rnd_alpha = normrnd(0,1,numage,1);
        for i=1:numage
            sample_varinv_tmp = 1./s2(:,n)';
            mu_alpha = sum((y_bayes(i,data_mat(i,:))' - betaa(i,n-1)*kappa(data_mat(i,:),n))'*diag(sample_varinv_tmp(data_mat(i,:))))/sum(sample_varinv_tmp(data_mat(i,:)));
            sigma_alpha = 1/sum(sample_varinv_tmp(data_mat(i,:)));
            alphaa(i,n) = sqrt(sigma_alpha)*rnd_alpha(i) + mu_alpha;
        end
    end
    
    % beta
    if polyb == 1
        % if beta is a cubic spline
        mu_tmp = 0;
        sigmainv_tmp = 0;
        for i = 1 : length(data_row)
            sample_varinv_tmp = 1/s2(data_col(i),n);
            mu_tmp = mu_tmp + polyx(data_row(i),:)*kappa(data_col(i),n) ...
                *(y_bayes(data_row(i),data_col(i))-alphaa(data_row(i),n))*sample_varinv_tmp;
            sigmainv_tmp = sigmainv_tmp + polyx(data_row(i),:)'*polyx(data_row(i),:)*kappa(data_col(i),n)^2 ...
                *sample_varinv_tmp;
        end
        [v,d] = eig(sigmainv_tmp);
        sigma_beta = v*diag(1./diag(d))*v';
        mu_beta = sigma_beta*mu_tmp';
        betaa(:,n) = polyx*mvnrnd(mu_beta,sigma_beta)';
    elseif polyb == 0
        % if beta is not smoothed
        rnd_beta = normrnd(0,1,numage,1);
        for i=1:numage
            mu_beta = sum(((y_bayes(i,data_mat(i,:)) - alphaa(i,n))'.*kappa(data_mat(i,:),n))...
                ./(s2(data_mat(i,:),n)'))...
                /sum(kappa(data_mat(i,:),n).^2./s2(data_mat(i,:),n)');
            sigma_beta = 1/sum(kappa(data_mat(i,:),n).^2./s2(data_mat(i,:),n)');
            betaa(i,n) = sqrt(sigma_beta)*rnd_beta(i) + mu_beta;
        end
    end
 
    % mu    
    mu(n) = normrnd((kappa(numyear,n)-kappa(1,n))/(numyear-1),sqrt(sigma2(n-1)/(numyear-1)));
    
    % sigma^2
    sigma2(n) = 1/gamrnd((numyear-1)/2,2/sum((kappa(2:numyear,n)-kappa(1:numyear-1,n)-mu(n)).^2));
     
    %% Apply Constraints
    sumbeta = sum(betaa(:,n));
    meankappa =  mean(kappa(:,n));
    kappa(:,n) = sumbeta*(kappa(:,n) - meankappa);
    alphaa(:,n) = alphaa(:,n) + betaa(:,n)*meankappa;
    betaa(:,n) = betaa(:,n)/sumbeta;
    
end
 
%% Burning and skipping
alpha_final = alphaa(:,numburn+1:numskip:end);
beta_final = betaa(:,numburn+1:numskip:end);
kappa_final = kappa(:,numburn+1:numskip:end);
mu_final = mu(numburn+1:numskip:end);
s2_final = s2(:,numburn+1:numskip:end);
sigma2_final = sigma2(numburn+1:numskip:end); 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% End of estimation
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Figure - log mortality rate
figure
subplot(1,3,1)
hold on
title('ln m_{x,2010}','FontWeight','Normal')
ylabel('Value')
xlabel('Age (x)')
plot(y_org(:,end-4),'ko')
xlim([1 100])
set(gca,'XTick',[1 21 41 61 81 100])
set(gca,'XTickLabel',{'0','20','40','60','80','99'})   
ylim([-9.5 -0.5])

subplot(1,3,2)
hold on
title('ln m_{x,2005}','FontWeight','Normal')
ylabel('Value')
xlabel('Age (x)')
plot(y_org(:,end-9),'ko')
xlim([1 100])
set(gca,'XTick',[1 21 41 61 81 100])
set(gca,'XTickLabel',{'0','20','40','60','80','99'})   
ylim([-9.5 -0.5])

subplot(1,3,3)
hold on
title('ln m_{x,2014}','FontWeight','Normal')
ylabel('Value')
xlabel('Age (x)')
plot(y_org(:,end),'ko')
xlim([1 100])
set(gca,'XTick',[1 21 41 61 81 100])
set(gca,'XTickLabel',{'0','20','40','60','80','99'})   
ylim([-9.5 -0.5])

%% Figure - alpha, beta and kappa
figure
subplot(1,3,1)
hold on
title('\alpha_x','FontWeight','Normal')
fanchart(alpha_final',0.1)
xlabel('Age (x)')
ylabel('Value')
xlim([1 100])
set(gca,'XTick',[1 21 41 61 81 100])
set(gca,'XTickLabel',{'0','20','40','60','80','99'})   
ylim([-8 -1])

subplot(1,3,2)
hold on
title('\beta_x','FontWeight','Normal')
fanchart(beta_final',0.1) 
xlabel('Age (x)')
ylabel('Value')
xlim([1 100])
set(gca,'XTick',[1 21 41 61 81 100])
set(gca,'XTickLabel',{'0','20','40','60','80','99'})
ylim([-0.01 0.04]) 

subplot(1,3,3)
hold on
title('\kappa_t','FontWeight','Normal')
fanchart(kappa_final',0.1)
xlabel('Year (t)')
ylabel('Value')
xlim([1 34])
set(gca,'XTick',[1 10 20 30])
set(gca,'XTickLabel',{'1981', '1990','2000','2010'})
ylim([-50 40])

%% Figure - s^2
figure
hold on
title('s^2','FontWeight','Normal')
histogram(s2_final(year_sample01(1),:),30,'Normalization','probability','EdgeAlpha',0.1)
histogram(s2_final(year_sample1(1),:),30,'Normalization','probability','EdgeAlpha',0.1)
histogram(s2_final(year_census(1),:),30,'Normalization','probability','EdgeAlpha',0.1)
xlabel('Value')
xlim([0 0.14])
ylim([0 0.14])
legend('s^2_{0.1%}','s^2_{1%}','s^2_{c}','Location','best')

%% Figure - mu and sigma^2
figure
subplot(1,2,2)
hold on 
title('\sigma^2','FontWeight','Normal')
histogram(sigma2_final,50,'Normalization','probability')
xlabel('Value')
xlim([0 100])

subplot(1,2,1)
hold on
title('\mu','FontWeight','Normal')
histogram(mu_final,50,'Normalization','probability')
xlabel('Value')
xlim([-6 2])

%% Figure - Prediction
% Allocate space
kappa_pred = zeros(numpred,numsam);
ycen_pred = zeros(numage,numpred,numsam);
ysam01_pred = zeros(numage,numpred,numsam);
ysam1_pred = zeros(numage,numpred,numsam);

% Predict future mortality rates
for n = 1:numsam  
    errork = normrnd(zeros(numpred,1),sqrt(sigma2_final(n)));
    errors = mvnrnd(zeros(numpred,numage),eye(numage)*s2_final(year_census(end),n));
    errors1 = mvnrnd(zeros(numpred,numage),eye(numage)*s2_final(year_sample1(end),n));
    errors01 = mvnrnd(zeros(numpred,numage),eye(numage)*s2_final(year_sample01(end),n));

    kappa_pred(1,n) = kappa_final(numyear,n)+ mu_final(n) + errork(1);
    ycen_pred(:,1,n) = alpha_final(:,n) + beta_final(:,n)*kappa_pred(1,n) + errors(1,:)';
    ysam01_pred(:,1,n) = alpha_final(:,n) + beta_final(:,n)*kappa_pred(1,n) + errors01(1,:)';
    ysam1_pred(:,1,n) = alpha_final(:,n) + beta_final(:,n)*kappa_pred(1,n) + errors1(1,:)';
    for j=2:numpred
        kappa_pred(j,n) = kappa_pred(j-1,n) + mu_final(n) + errork(j);
        ycen_pred(:,j,n) = alpha_final(:,n) + beta_final(:,n)*kappa_pred(j,n) + errors(j,:)';
        ysam01_pred(:,j,n) = alpha_final(:,n) + beta_final(:,n)*kappa_pred(j,n) + errors01(j,:)';
        ysam1_pred(:,j,n) = alpha_final(:,n) + beta_final(:,n)*kappa_pred(j,n) + errors1(j,:)';
    end
end

figure
subplot(4,3,1)
hold on
title('ln m_{0,t}','FontWeight','Normal')
plot(year_census,log(m_org(1,year_census)),'ok')
plot(year_sample1,log(m_org(1,year_sample1)),'sk')
plot(year_sample01,log(m_org(1,year_sample01)),'xk')
cichart([NaN(numsam,size(m_org,2)) squeeze(ycen_pred(1,:,:))'],'k')
cichart([NaN(numsam,size(m_org,2)) squeeze(ysam1_pred(1,:,:))'],'k--')
cichart([NaN(numsam,size(m_org,2)) squeeze(ysam01_pred(1,:,:))'],'k.')
plot((size(m_org,2)+1):size(m_org,2)+numpred,mean(ycen_pred(1,:,:),3),'k')
xlim([1 numyear+numpred])
xlabel('Year (t)')
ylabel('Value')
set(gca,'XTick',[1 20 40 60])
set(gca,'XTickLabel',{'1981','2000','2020','2040'})
ylim([-12 -2])

for i = 1 : 9
    subplot(4,3,i+3)
    hold on
    title(['ln m_{' num2str(i*10) ',t}'],'FontWeight','Normal')
    plot(year_census,log(m_org(i*10+1,year_census)),'ok')
    plot(year_sample1,log(m_org(i*10+1,year_sample1)),'sk')
    plot(year_sample01,log(m_org(i*10+1,year_sample01)),'xk')
    cichart([NaN(numsam,size(m_org,2)) squeeze(ycen_pred(i*10+1,:,:))'],'k')
    cichart([NaN(numsam,size(m_org,2)) squeeze(ysam1_pred(i*10+1,:,:))'],'k--')
    cichart([NaN(numsam,size(m_org,2)) squeeze(ysam01_pred(i*10+1,:,:))'],'k.')
    plot((size(m_org,2)+1):size(m_org,2)+numpred,mean(ycen_pred(i*10+1,:,:),3),'k')
    xlim([1 numyear+numpred])
    xlabel('Year (t)')
    ylabel('Value')
    set(gca,'XTick',[1 20 40 60])
    set(gca,'XTickLabel',{'1981','2000','2020','2040'})
    if i == 1
        ylim([-10.5 -6.5])
    elseif i == 2
        ylim([-9.8 -5.9])
    elseif i == 3
        ylim([-8.6 -5.8])
    elseif i == 4
        ylim([-7.8 -5.2]) 
    elseif i == 5
        ylim([-6.9 -4.6])
    elseif i == 6
        ylim([-6.4 -3.6])
    elseif i == 7
        ylim([-5.4 -2.6])
    elseif i == 8
        ylim([-4.2 -1.8])
    elseif i == 9
        ylim([-3.1 -0.9])
    end
end

%% Figure - Prediction without parameter risk
% Allocate space
kappa_nopu = zeros(numpred,numsam);
ycen_nopu = zeros(numage,numpred,numsam);
kappa_pred = zeros(numpred,numsam);
ycen_pred = zeros(numage,numpred,numsam);

% Calculate posterior means
alpha_mean = mean(alpha_final,2);
beta_mean = mean(beta_final,2);
kappa_mean = mean(kappa_final,2);
sigma2_mean = mean(sigma2_final);
s2_mean = mean(s2_final,2);
mu_mean = mean(mu_final);

% Predict future mortality rates
for n = 1:numsam  
    errork = normrnd(zeros(numpred,1),sqrt(sigma2_mean));
    errors = mvnrnd(zeros(numpred,numage),eye(numage)*s2_mean(1));
    errork0 = normrnd(zeros(numpred,1),sqrt(sigma2_final(n)));
    errors0 = mvnrnd(zeros(numpred,numage),eye(numage)*s2_final(1,n));

    kappa_nopu(1,n) = kappa_mean(numyear)+ mu_mean + errork(1);   
    ycen_nopu(:,1,n) = alpha_mean + beta_mean*kappa_nopu(1,n) + errors(1,:)';    
    kappa_pred(1,n) = kappa_final(numyear,n)+ mu_final(n) + errork0(1);
    ycen_pred(:,1,n) = alpha_final(:,n) + beta_final(:,n)*kappa_pred(1,n) + errors0(1,:)';
    
    for j=2:numpred
        kappa_nopu(j,n) = kappa_nopu(j-1,n) + mu_mean + errork(j);
        ycen_nopu(:,j,n) = alpha_mean + beta_mean*kappa_nopu(j,n) + errors(j,:)';
        kappa_pred(j,n) = kappa_pred(j-1,n) + mu_final(n) + errork0(j);
        ycen_pred(:,j,n) = alpha_final(:,n) + beta_final(:,n)*kappa_pred(j,n) + errors0(j,:)';
    end
end

figure
subplot(4,3,1)
hold on
title('ln m_{0,t}','FontWeight','Normal')
cichart(squeeze(ycen_pred(1,:,:))','k-');
cichart(squeeze(ycen_nopu(1,:,:))','b--');
xlim([1 numpred])
xlabel('Year (t)')
ylabel('Value')
set(gca,'XTick',[1 11 21 31])
set(gca,'XTickLabel',{'2015','2025', '2035','2045'})
    
for i = 1 : 9
    subplot(4,3,i+3)
    hold on
    title(['ln m_{' num2str(i*10) ',t}'],'FontWeight','Normal')
    cichart(squeeze(ycen_pred(i*10+1,:,:))','k-');
    cichart(squeeze(ycen_nopu(i*10+1,:,:))','b--');
    xlim([1 numpred])
    xlabel('Year (t)')
    ylabel('Value')
    set(gca,'XTick',[1 11 21 31])
    set(gca,'XTickLabel',{'2015','2025', '2035','2045'})
end

%% Local functions for generating figures
% Function 1: fan chart
% This function generates a fan chart.
function []=fanchart(fore,g)
    % Compute quantiles
    quant=(5:5:95)'/100;
    mquant=quantile(fore,quant);
    midspot = ceil(length(quant)/2);
    matm = [mquant(1,:) ; diff(mquant)];
    % Generate figure
    r = 0.1; b = 0.1;
    cha = 0.05;
    h=area(matm',-1000,'EdgeAlpha',0);
    set(h,'LineStyle','none');
    set(h(1),'FaceColor',[1 1 1],'Visible','off');
    for i = 2 : midspot
    set(h(midspot+2-i),'FaceColor',[r+cha*(i-1) g+cha*(i-1)  b+cha*(i-1)]);
    end
    for i = midspot+1 : length(quant)
    set(h(i),'FaceColor',[r+cha*(i-midspot) g+cha*(i-midspot)  b+cha*(i-midspot)]);
    end
end

% Function 2: CI chart
% This function generates a CI chart.
function [h]=cichart(fore,g)
    % Compute quantiles and draw lines
    quant=[5 95]'/100;
    mquant=quantile(fore,quant);
    h=plot(mquant',g);
end
