%%%%%%%%%%%%%
% BEKKsim.m %
%%%%%%%%%%%%%

% Objective:
% - To simulate cumulative log returns from the S&P600 and S&P500
% - Assumed model is a bivariate BEKK-GARCH model
% - Model parameter values are taken from Ng, A.C.Y. and Li, J.S.-H. (2013) 
% Pricing and Hedging Variable Annuity Guarantees with Multi-Asset 
% Stochastic Investment Models. North American Actuarial Journal, 17, 41-62

% Inputs:
% (1) steps - number of time steps (months) ahead
% (2) paths - number of sample paths demanded
% (3) r - continuously compounded risk-free interest rate per month

% Outputs:
% cum_yp: simulated cumulated log returns under the real-world (P) measure
% cum_yq: simulated cumulated log returns under the risk-neutral (Q) measure

function [cum_yp, cum_yq] = BEKKsim(steps, paths, r)

% Model is BEKK-GARCH
% Model parameters under P-measure (estimated from historical returns)
% Taken from page 6 of Ng and Li (2013)

C = [0.0071184; 0.0047536]; % drift in conditional mean equation

% drift in the equation for H_t 
Ome = [0.017040593 0; 0.011807674 0.001882236] * [0.017040593 0; 0.011807674 0.001882236]'; 

% A matrix in the equation for H_t 
A = [0.381059834  -0.004046219 
    -0.003142479   0.393367842];

% B matrix in the equation for H_t 
B = [0.901775237 -0.012810967
    -0.021156602  0.913774357];

% % The required pre-sample observations, innovations, and conditional
% % variances (calculated from the fitted model)

pre_a=[0.098635245; 0.123422301];

pre_H=[0.005633967  0.003956135
       0.003956135  0.0029351];

% Initialization
a=zeros(2,steps);
yp=zeros(2,steps,paths);
yq=zeros(2,steps,paths);
H=zeros(2,2,steps);

% Main loop

for j=1:paths
    
    % First step
    H(:,:,1) = Ome + A*(pre_a*pre_a')*A' + B* pre_H * B';   
    a(:,1) = chol(H(:,:,1),'lower')*randn(2,1);    
    yp(:,1,j) = C + a(:,1);
    yq(1,1,j) = r - 0.5 * H(1,1,1) + a(1,1);
    yq(2,1,j) = r - 0.5 * H(2,2,1) + a(2,1);   
    
    % 2nd step and onwards
    for i=2:steps
        H(:,:,i) = Ome + A * a(:,i-1)*a(:,i-1)'*A' + B * H(:,:,i-1)*B';        
        a(:,i) = chol(H(:,:,i),'lower')*randn(2,1);
        yp(:,i,j) = C + a(:,i);   
        yq(1,i,j) = r - 0.5 * H(1,1,i) + a(1,i);  
        yq(2,i,j) = r - 0.5 * H(2,2,i) + a(2,i);  
    end
    
end

% Calculating culmulative returns
cum_yp=exp(cumsum(yp,2));
cum_yq=exp(cumsum(yq,2));