%%%%%%%%%%%%
% CCCsim.m %
%%%%%%%%%%%%

% Objective:
% - To simulate cumulative log returns from the S&P600 and S&P500
% - Assumed model is a bivariate CCC-GARCH model
% - Model parameter values are taken from Ng, A.C.Y. and Li, J.S.-H. (2013) 
% Pricing and Hedging Variable Annuity Guarantees with Multi-Asset 
% Stochastic Investment Models. North American Actuarial Journal, 17, 41-62

% Inputs:
% (1) steps - number of time steps (months) ahead
% (2) paths - number of sample paths demanded
% (3) r - continuously compounded risk-free interest rate per month

% Outputs:
% cum_yp: simulated cumulated log returns under the real-world (P) measure
% cum_yq: simulated cumulated log returns under the risk-neutral (Q) measure


function [cum_yp,cum_yq] = CCCsim(steps,paths,r)

% Model is CCC
% Model parameters under P-measure (estimated from historical returns)
% Taken from page 6 of Ng and Li (2013)

% drift in conditional mean equation
C = [0.0071184; 
     0.0047536]; 

% drift in the equation for H_t
K = [0.000386852;
     0.0000659172]; 

% Simiplified matrices in the equation for H_t

A = [0.105090169 0
     0           0.191529717];

B = [0.786626336 0
     0           0.803289537];

P = [1           0.80102678
     0.80102678  1          ];
    

% The required pre-sample observations, innovations, and conditional
% variances (calculated from the fitted model)
 
pre_a=[0.13444406; 0.092549156];
pre_h=[0.00479953; 0.00337287];

% Initialization
a=zeros(2,steps);
yp=zeros(2,steps,paths);
yq=zeros(2,steps,paths);
h=zeros(2,steps);
H=zeros(2,2,steps);

% Main loop

for j=1:paths
    
    % First step
    h(:,1) = K + A*(pre_a.*pre_a) + B* pre_h ;
    H(:,:,1) = diag(sqrt(pre_h))*P*diag(sqrt(pre_h));   
    a(:,1) = chol(H(:,:,1),'lower')*randn(2,1);    
    yp(:,1,j) = C + a(:,1);
    %yq(:,1,j) = r*ones(2,1) - C - 0.5 * h(:,1) + a(:,1);
    yq(:,1,j) = r*ones(2,1) - 0.5 * h(:,1) + a(:,1);
    
    % 2nd step and onwards
    for i=2:steps
        h(:,i) = K + A * (a(:,i-1).*a(:,i-1)) + B * h(:,i-1) ;        
        H(:,:,i) = diag(sqrt(h(:,i))) * P * diag(sqrt(h(:,i)));        
        a(:,i) = chol(H(:,:,i),'lower')*randn(2,1);
        yp(:,i,j) = C + a(:,i);   
        %yq(:,i,j) = r*ones(2,1) - C - 0.5 * h(:,i) + a(:,i);
        yq(:,i,j) = r*ones(2,1) - 0.5 * h(:,i) + a(:,i);
    end
    
end

% Calculating culmulative returns
cum_yp=exp(cumsum(yp,2));
cum_yq=exp(cumsum(yq,2));