%%%%%%%%%%%%
% DCCsim.m %
%%%%%%%%%%%%

% Objective:
% - To simulate cumulative log returns from the S&P600 and S&P500
% - Assumed model is a bivariate DCC-GARCH model
% - Model parameter values are taken from Ng, A.C.Y. and Li, J.S.-H. (2013) 
% Pricing and Hedging Variable Annuity Guarantees with Multi-Asset 
% Stochastic Investment Models. North American Actuarial Journal, 17, 41-62

% Inputs:
% (1) steps - number of time steps (months) ahead
% (2) paths - number of sample paths demanded
% (3) r - continuously compounded risk-free interest rate per month

% Outputs:
% cum_yp: simulated cumulated log returns under the real-world (P) measure
% cum_yq: simulated cumulated log returns under the risk-neutral (Q) measure

function [cum_yp,cum_yq] = DCCsim(steps, paths, r)

% Model is DCC
% Model parameters under P-measure (estimated from historical returns)
% Taken from page 6 of Ng and Li (2013)

% drift in conditional mean equation
C = [0.0071184;
     0.0047536];    

% drift in the equation for H_t
K = [0.000386852;
     0.0000659172]; 

% Other parameters in the euqation for H_t
A = [0.105090169 0
     0           0.191529717];

B = [0.786626336 0
     0           0.803289537];

alp = 0.105842397;
bet = 0.894155603;

S = [1    0.818392
     0.818392   1];

% % The required pre-sample observations, innovations, and conditional
% % variances (calculated from the fitted model)

pre_a=[0.156835236;   
       0.05502056];

pre_h=[0.00479953; 0.00337287];

preH = [0.00479953 0.003915747
        0.003915747 0.00337287];

pre_Q = [1.403955965 1.276498813
         1.276498813 1.22533985];

% Initialization
a=zeros(2,steps);
yp=zeros(2,steps,paths);
yq=zeros(2,steps,paths);
h=zeros(2,steps);
H=zeros(2,2,steps);
Q = zeros(2,2,steps);
P = zeros(2,2,steps);

% Main loop

for j=1:paths
    
    % First step
    h(:,1) = K + B* pre_h + A*(pre_a.*pre_a);
    Q(:,:,1) = (1 - alp - bet) * S + alp * pre_a * pre_a' + bet * pre_Q;
    H(:,:,1) = preH;   
    a(:,1) = chol(H(:,:,1),'lower')*randn(2,1);    
    yp(:,1,j) = C + a(:,1);
    yq(:,1,j) = r*ones(2,1) - 0.5 * h(:,1) + a(:,1);
    
    % 2nd step and onwards
    for i=2:steps
        h(:,i) = K + B * h(:,i-1) + A * (a(:,i-1).*a(:,i-1));        
        Q(:,:,i) = (1 - alp - bet) * S + alp * a(:, i-1) * a(:, i-1)' + bet * Q(:,:,i-1);
        rho = Q(1,2,i)/(Q(1,1,i)*Q(2,2,i))^0.5;
        P(:,:,i) = [1  rho; rho 1];
        H(:,:,i) = diag(sqrt(h(:,i))) * P(:,:,i) * diag(sqrt(h(:,i)));
        a(:,i) = chol(H(:,:,i),'lower')*randn(2,1);
        yp(:,i,j) = C + a(:,i);   
        yq(:,i,j) = r*ones(2,1) - 0.5 * h(:,i) + a(:,i);
    end
    
end

% Calculating culmulative returns
cum_yp=exp(cumsum(yp,2));
cum_yq=exp(cumsum(yq,2));