%%%%%%%%%%%%%%%%%%
% Illustration.m %
%%%%%%%%%%%%%%%%%%


% Objective: 
% - To graph sample paths generated by a bi-variate stochastic investment
% model, in both real-world (P) and risk-neutral (Q) measures.
% - The two variables under consideration are log-returns on S&P500 and S&P
% 600 stock indexes
% - Model choices are BEKK-GARCH, CCC-GARCH, DCC-GARCH, and
% Regime-switching log-normal
% - Model parameter values are taken from Ng, A.C.Y. and Li, J.S.-H. (2013) 
% Pricing and Hedging Variable Annuity Guarantees with Multi-Asset 
% Stochastic Investment Models. North American Actuarial Journal, 17, 41-62


% Inputs:
% (1) ts_model - the chosen bi-variate stochastic investment model
% "BEKK": BEKK-GARCH model
% "CCC": CCC-GARCH model
% "DCC": DCC-GARCH model
% "RSLN": Regime-switching-log-normal (RSLN-2)
%  model
% (2) steps - number of time steps (months) ahead
% (3) paths - number of sample paths demanded
% (4) r - continuously compounded risk-free interest rate per month


% Output:
% A diagram of sample paths
% left panel: simulated cumulative S&P 600 returns
% right panel: simulated cumulative S&P 500 returns
% green lines: sample paths in the real-world probability measure
% blue lines: sample paths in the risk-neutral probability measure


% Example:
% Illustration("BEKK", 30, 100, 0.02/12);
% This command line generates 100 sample paths from the BEKK-GARCH model,
% with a horizon of 30 months, given a continuously compounded risk-free
% interest rate of 0.02/12 per month (i.e., 2% per annum).

function Illustration(ts_model, steps, paths, r)

% generate sample paths based on the chosen model
if ts_model == "BEKK"
    [cum_yp,cum_yq] = BEKKsim(steps, paths, r);
elseif ts_model == "CCC"
    [cum_yp,cum_yq] = CCCsim(steps, paths, r);
elseif ts_model == "DCC"
    [cum_yp,cum_yq] = DCCsim(steps, paths, r);
elseif ts_model == "RSLN"
    [cum_yp,cum_yq] = RSLN2simQ(steps, paths, r);
end

% showing the standard deviations of simulated returns 
disp(['Standrd deviation of S&P 600 cumulative returns in ' num2str(steps) ' months: ' num2str(std(cum_yp(1, end, :)))]);
disp(['Standrd deviation of S&P 500 cumulative returns in ' num2str(steps) ' months: ' num2str(std(cum_yp(2, end, :)))]);

% consolidating results
p1 = squeeze(cum_yp(1,:,:));
p2 = squeeze(cum_yp(2,:,:));
q1 = squeeze(cum_yq(1,:,:));
q2 = squeeze(cum_yq(2,:,:));

% plotting sample paths
subplot(1,2,1)
plot(p1,'color','g'); hold on; plot(q1,'color','b'); xlabel('Time in months'); ylabel('Cumulative return'); title('S&P 600'); hold off;
subplot(1,2,2)
plot(p2,'color','g'); hold on; plot(q2,'color','b'); xlabel('Time in months'); ylabel('Cumulative return'); title('S&P 500'); hold off;

% comment on the plots
disp('Green lines: sample paths in the real-world probablity measure')
disp('Blue lines: sample paths in the risk-neutral probablity measure')