%%%%%%%%%%%%%%
% RSLN2sim.m %
%%%%%%%%%%%%%%

% Objective:
% - To simulate cumulative log returns from the S&P600 and S&P500
% - Assumed model is a bivariate regime-switching log-normal (RSLN2) model
% - Model parameter values are taken from Ng, A.C.Y. and Li, J.S.-H. (2013) 
% Pricing and Hedging Variable Annuity Guarantees with Multi-Asset 
% Stochastic Investment Models. North American Actuarial Journal, 17, 41-62

% Inputs:
% (1) steps - number of time steps (months) ahead
% (2) paths - number of sample paths demanded
% (3) r - continuously compounded risk-free interest rate per month

% Outputs:
% cum_yp: simulated cumulated log returns under the real-world (P) measure
% cum_yq: simulated cumulated log returns under the risk-neutral (Q) measure
% mcstate: states of the Markov-chain over the calibration window


function [cum_yp, cum_yq, mcstate] = RSLN2sim(steps, paths, r)

% Model is RSLN2
% Model is DCC
% Model parameters under P-measure (estimated from historical returns)
% Taken from page 7 of Ng and Li (2013)

% parameters in regime 1
mu1 = [0.01433812; 0.01143726];
sx1	= 0.03657781;
sy1	= 0.02439095;

% parameters in regime 2
mu2	= [0.00173456; -0.00023164];
sx2	= 0.06788845;
sy2	= 0.05642218;

% corrleation in both states
rho	= 0.81150050;

% transition matrix
P = [0.96475197 0.03524803
0.02904235 0.97095765];

% Initialization

mcstate = ones(steps, paths);  % state of the markov chain 
yp=zeros(2,steps,paths);
yq=zeros(2,steps,paths);

decom = [1 0; rho (1 - rho^2)^0.5];
cho1 = [sx1 0; 0 sy1] * decom;
cho2 = [sx2 0; 0 sy2] * decom;
pi1 = P(2,1)/(P(1,2)+P(2,1));   % stationary probability for state 1
h1 = 0.5*[sx1^2; sy1^2];
h2 = 0.5*[sx2^2; sy2^2];

% Main loop

for j=1:paths
    
    % first simulate the states
    
    if rand(1) > pi1
        mcstate(1,j) = 2;
    end
    
    for i = 2: steps
        if mcstate(i-1,j) == 1
            if rand(1) > P(1, 1)
                mcstate(i, j) = 2;
            end
        else 
            if rand(1) < P(2, 2)
                mcstate(i, j) = 2;
            end
        end
    end
    
    % then simulate the returns
    
    for i = 1: steps
        if mcstate(i, j) == 1
            yp(:,i,j) = mu1 + cho1*randn(2,1);
            yq(:,i,j) = r*ones(2,1) - h1 + cho1*randn(2,1);
        else
            yp(:,i,j) = mu2 + cho2*randn(2,1);
            yq(:,i,j) = r*ones(2,1) - h2 + cho2*randn(2,1);
        end
    end
    
end

% Calculating culmulative returns
cum_yp=exp(cumsum(yp,2));
cum_yq=exp(cumsum(yq,2));